package edu.uwyo.cs.cell.client;

import java.util.*;
import java.rmi.*;

import org.apache.commons.logging.*;

import org.apache.axis.message.addressing.*;

import org.globus.wsrf.*;
import org.globus.wsrf.core.notification.*;
import org.globus.wsrf.core.notification.service.*;
import org.oasis.wsn.*;
import org.oasis.wsrf.lifetime.*;

import edu.uwyo.cs.cell.impl.*;
import edu.uwyo.cs.cell.stubs.*;
import edu.uwyo.cs.cell.stubs.service.*;
import edu.uwyo.cs.cell.factory.stubs.*;
import edu.uwyo.cs.cell.factory.stubs.service.*;

public class CellClient implements NotifyCallback {
    static final Log logger = LogFactory.getLog (CellClient.class);

    private static final String serviceURIPostfix = ":8080/wsrf/services/CellService";
    private static final String factoryURIPostfix = ":8080/wsrf/services/CellFactoryService";

    private int NUM_CELLS = 51;
    private int NUM_STEPS = 1;

    private CellInfo[] cell;

    private int numFinished = 0;
    private boolean done = false;

    NotificationConsumerManager consumer;

    public static void main (String[] args) {
	CellClient client = new CellClient ();

	client.initialize (args);
    }

    public void initialize (String[] args) {
	String factoryURI = "http://localhost" + factoryURIPostfix;

	switch (args.length) {
	default:
	    System.out.println ("Usage: CellClient [ num_cells ] [ num_steps ]");
	    return;
	case 2:
	    NUM_STEPS = Integer.parseInt (args[1]);
	    /* nobreak; */
	case 1:
	    NUM_CELLS = Integer.parseInt (args[0]);
	    /* nobreak; */
	case 0:
	    break;
	}

	cell = new CellInfo[NUM_CELLS];

	try {
	    FactoryServiceAddressingLocator factoryLocator = new FactoryServiceAddressingLocator ();
	    CellServiceAddressingLocator cellLocator = new CellServiceAddressingLocator ();

	    EndpointReferenceType factoryEndpoint = new EndpointReferenceType ();
	    factoryEndpoint.setAddress (new Address (factoryURI));
	    FactoryPortType cellFactory = factoryLocator.getFactoryPortTypePort (factoryEndpoint);

	    for (int i=0; i<NUM_CELLS; i++) {
		logger.info ("Creating cell: " + i);
		CreateResourceResponse createResponse = cellFactory.createResource (new CreateResource());
		cell[i] = new CellInfo ();
		cell[i].endpoint = createResponse.getEndpointReference ();
		cell[i].cell = cellLocator.getCellPortTypePort (cell[i].endpoint);
		cell[i].cell.setPosition (i);
		cell[i].value = 0;
	    }
	    cell[NUM_CELLS/2].value = 1;

	    print_cells ();

	    // setupNotification ();

	    for (int i=0; i<NUM_CELLS; i++) {
		logger.info ("Evolving cell: " + i);
		cell[i].cell.evolve (new Evolve (( (i-1) >= 0 ? cell[i-1].endpoint : null),
						 new Integer (NUM_STEPS),
						 new Integer (cell[i].value),
						 new Integer (i),
						 ( (i+1) < NUM_CELLS ? cell[i+1].endpoint : null)));
	    }
	    while (!done)
		Thread.sleep (5000);
	    System.out.println ("Exiting!");
	}
	catch (Exception e) {
	    e.printStackTrace ();
	}
    }

    private void setupNotification () {
	try {
	    consumer = NotificationConsumerManager.getInstance();
	    consumer.startListening();
	    EndpointReferenceType consumerEPR = consumer
		.createNotificationConsumer(this);

	    Subscribe request = new Subscribe();

	    request.setUseNotify(Boolean.TRUE);
	    request.setConsumerReference(consumerEPR);

	    TopicExpressionType topicExpression = new TopicExpressionType();
	    topicExpression.setDialect(WSNConstants.SIMPLE_TOPIC_DIALECT);
	    topicExpression.setValue(CellConstants.TOPIC_FINISHED);
	    request.setTopicExpression(topicExpression);

	    WSBaseNotificationServiceAddressingLocator notifLocator = new WSBaseNotificationServiceAddressingLocator();

	    for (int i=0; i<NUM_CELLS; i++) {
		System.out.println ("Registering with port: " + i);
		NotificationProducer producerPort = notifLocator
		    .getNotificationProducerPort(cell[i].endpoint);
		producerPort.subscribe(request);
	    }
	}
	catch (Exception e) {
	    e.printStackTrace();
	}
    }

    public void deliver (List topicPath, EndpointReferenceType producer, Object message) {
	try {
	    FinishedNotificationMessageWrapperType wrp =
		(FinishedNotificationMessageWrapperType) message;
	    FinishedNotificationMessageType fini = wrp.getFinishedNotificationMessage();
	    int pos = fini.getPosition ();
	    int value = fini.getValue ();
	    cell[pos].value = value;
	    System.out.println ("Callback: pos=" + pos + " value=" + value);
	}
	catch (Exception e) {
	    e.printStackTrace();
	}
    }

    public synchronized void finishProcessing () {
	try {
	    numFinished ++;
	    if (numFinished < NUM_CELLS)
		return;

	    print_cells ();
	    System.out.println ("Finished!");

	    for (int i=0; i<NUM_CELLS; i++)
		cell[i].cell.destroy (new Destroy());

	    consumer.stopListening ();

	    done = true;
	}
	catch (Exception e) {
	    e.printStackTrace();
	}
    }

    public void print_cells () {
	for (int i=0; i<NUM_CELLS; i++)
	    if (cell[i].value == 0)
		System.out.print (".");
	    else
		System.out.print ("*");
	System.out.println ();
    }

    private static class CellInfo {
	public EndpointReferenceType endpoint;
	public CellPortType cell;
	public int value;
    }

}

