package edu.uwyo.cs.cell.impl;

import java.rmi.RemoteException;

import org.globus.wsrf.*;
import org.apache.axis.message.addressing.*;
import org.apache.commons.logging.*;

import edu.uwyo.cs.cell.stubs.*;
import edu.uwyo.cs.cell.stubs.service.*;

public class CellService {

    static final Log logger = LogFactory.getLog (CellService.class);

    private CellResource getResource() throws RemoteException {
	Object resource = null;
	try {
	    resource = ResourceContext.getResourceContext().getResource();
	} catch (Exception e) {
	    throw new RemoteException("Error looking up resource", e);
	}

	CellResource cellResource = (CellResource) resource;
	return cellResource;
    }

    public EvolveResponse evolve (Evolve params) throws RemoteException {
	CellResource cellResource = getResource();

	logger.trace ("Evolve:: cell: " + params.getPosition() + 
		     " NumSteps: " + params.getNumSteps());

	cellResource.setPosition (((Integer)params.getPosition()).intValue());
	cellResource.setValue (((Integer)params.getOrigValue()).intValue());
	cellResource.setNumSteps (((Integer)params.getNumSteps()).intValue());
	cellResource.setLeftCellEPR (params.getLeftCell());
	cellResource.setRightCellEPR (params.getRightCell());
	broadcastAndStep ();

	return new EvolveResponse ();
    }

    public SetPositionResponse setPosition (int position)  throws RemoteException {
	logger.trace ("SetPosition:: cell: " + position);

	CellResource cellResource = getResource();
	cellResource.setPosition (position);

	return new SetPositionResponse ();
    }

    public SetLeftValueResponse setLeftValue (int left)  throws RemoteException {
	CellResource cellResource = getResource();

	logger.trace ("SetLeftValue:: cell: " + cellResource.getPosition() + 
		     " Left: " + left);

	if (cellResource.getLeftIsSet ()) {
	    cellResource.setPrevLeftValue (left);
	    cellResource.setPrevLeftIsSet (true);
	    logger.trace ("PushingLeftValue:: cell: " + cellResource.getPosition());
	}
	else {
	    cellResource.setLeftValue (left);
	    cellResource.setLeftIsSet (true);
	    step ();
	}
	
	return new SetLeftValueResponse ();
    }
    
    public SetRightValueResponse setRightValue (int right)  throws RemoteException {
	CellResource cellResource = getResource();

	logger.trace ("SetRightValue:: cell: " + cellResource.getPosition() + 
		     " Right: " + right);

	if (cellResource.getRightIsSet ()) {
	    cellResource.setPrevRightValue (right);
	    cellResource.setPrevRightIsSet (true);
	    logger.trace ("PushingRightValue:: cell: " + cellResource.getPosition());
	}
	else {
	    cellResource.setRightValue (right);
	    cellResource.setRightIsSet (true);
	    step ();
	}

	return new SetRightValueResponse ();
    }

    private void step ()  throws RemoteException {
	final CellResource cellResource = getResource ();
	Thread command = new Thread () {
		public void run () {
		    try {
			step (cellResource);
		    }
		    catch (Exception e) {
			logger.error ("Error performing step", e);
		    }
		}
	    };
	command.start ();
    }

    private synchronized void step (CellResource cellResource)  throws RemoteException {
	boolean doneStepping;

	do {
	    doneStepping = true;

	    // logger.trace ("Thinking About Stepping:: cell: " + cellResource.getPosition());

	    if (cellResource.getNumSteps() <= 0) {
		// logger.trace ("Not Stepping (out of steps):: cell: " + cellResource.getPosition());
		return;
	    }

	    if (cellResource.getLeftCellEPR() != null && !cellResource.getLeftIsSet()) {
		// logger.trace ("Not Stepping (wait on left):: cell: " + cellResource.getPosition());
		return;
	    }

	    if (cellResource.getRightCellEPR() != null && !cellResource.getRightIsSet()) {
		// logger.trace ("Not Stepping (wait on right):: cell: " + cellResource.getPosition());
		return;
	    }

	    cellResource.setNumSteps (cellResource.getNumSteps()-1);

	    logger.trace ("Stepping:: cell: " + cellResource.getPosition() + " stepsLeft: " + cellResource.getNumSteps());

	    cellResource.setValue (rule30 (cellResource.getLeftValue(),
					   cellResource.getValue(),
					   cellResource.getRightValue()));
	    if (cellResource.getPrevLeftIsSet ()) {
		cellResource.setLeftValue (cellResource.getPrevLeftValue());
		cellResource.setPrevLeftIsSet (false);
		doneStepping = false;
		logger.trace ("Reusing old left:: cell: " + cellResource.getPosition());
	    }
	    else
		cellResource.setLeftIsSet (false);
	    if (cellResource.getPrevRightIsSet ()) {
		cellResource.setRightValue (cellResource.getPrevRightValue());
		cellResource.setPrevRightIsSet (false);
		doneStepping = false;
		logger.trace ("Reusing old right:: cell: " + cellResource.getPosition());
	    }
	    else
		cellResource.setRightIsSet (false);
	    broadcast (cellResource);
	} while (!doneStepping);
    }

    private void broadcastAndStep ()  throws RemoteException {
	final CellResource cellResource = getResource ();
	Thread command = new Thread () {
		public void run () {
		    try {
			broadcast (cellResource);
			step (cellResource);
		    }
		    catch (Exception e) {
			logger.error ("Error performing broadcastAndStep", e);
		    }
		}
	    };
	command.start ();
    }

    private void broadcast ()  throws RemoteException {
	final CellResource cellResource = getResource ();
	Thread command = new Thread () {
		public void run () {
		    try {
			broadcast (cellResource);
		    }
		    catch (Exception e) {
			logger.error ("Error performing broadcast", e);
		    }
		}
	    };
	command.start ();
    }

    private synchronized void broadcast (CellResource cellResource)  throws RemoteException {

	logger.trace ("Broadcasting:: cell: " + cellResource.getPosition());

	try {
	    logger.trace ("Updating Neighbors:: cell: " + cellResource.getPosition());

	    CellServiceAddressingLocator cellLocator = new CellServiceAddressingLocator ();

	    EndpointReferenceType left = cellResource.getLeftCellEPR ();
	    if (left != null) {
		CellPortType leftNeighbor = cellLocator.getCellPortTypePort (left);
		leftNeighbor.setRightValue (cellResource.getValue());
	    }

	    EndpointReferenceType right = cellResource.getRightCellEPR ();
	    if (right != null) {
		CellPortType rightNeighbor = cellLocator.getCellPortTypePort (right);
		rightNeighbor.setLeftValue (cellResource.getValue());
	    }

	    if (cellResource.getNumSteps() <= 0) {
		logger.info ("Finished:: cell: " + cellResource.getPosition() + " value: " + cellResource.getValue());

		FinishedNotificationMessageType msg = new FinishedNotificationMessageType (cellResource.getPosition(), cellResource.getValue());
		FinishedNotificationMessageWrapperType wrp = 
		    new FinishedNotificationMessageWrapperType (msg);
		cellResource.getFinishedTopic().notify (wrp);
	    }
	}
	catch (Exception e) {
	    throw new RemoteException ("Exception raised while notifying neighbors", e);
	}
    }
    
    private int rule30 (int left, int cur, int right) {
	int next;

	if (left == 0)
	    if (cur == 0 && right == 0)
		next = 0;
	    else
		next = 1;
	else
	    if (cur == 0 && right == 0)
		next = 1;
	    else
		next = 0;

	return next;
    }

}
